<?php
/**
 * Email Verification Migration Script
 * Adds email verification columns to the users table
 */

require_once 'api/config.php';

echo "<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <title>Email Verification Migration</title>
    <style>
        body { font-family: Arial, sans-serif; max-width: 800px; margin: 50px auto; padding: 20px; background: #f5f5f5; }
        .status { background: white; padding: 20px; margin: 20px 0; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1); }
        .success { border-left: 4px solid #10b981; }
        .error { border-left: 4px solid #ef4444; }
        .warning { border-left: 4px solid #f59e0b; }
        h1 { color: #2c3e50; }
        code { background: #f8f9fa; padding: 2px 6px; border-radius: 4px; }
    </style>
</head>
<body>
    <h1>🔐 Email Verification Migration</h1>";

try {
    $pdo = getDBConnection();
    echo "<div class='status success'><strong>✓ Database Connection:</strong> Success</div>";

    // Check if columns already exist
    $stmt = $pdo->query("SHOW COLUMNS FROM users LIKE 'email_verified'");
    $emailVerifiedExists = $stmt->rowCount() > 0;

    if ($emailVerifiedExists) {
        echo "<div class='status warning'>";
        echo "<strong>ℹ️ Migration Already Completed</strong><br>";
        echo "Email verification columns already exist.";
        echo "</div>";
    } else {
        // Add email_verified column
        $pdo->exec("ALTER TABLE users ADD COLUMN email_verified BOOLEAN DEFAULT FALSE");
        echo "<div class='status success'><strong>✓</strong> Added <code>email_verified</code> column</div>";

        // Add verification_token column
        $pdo->exec("ALTER TABLE users ADD COLUMN verification_token VARCHAR(255) DEFAULT NULL");
        echo "<div class='status success'><strong>✓</strong> Added <code>verification_token</code> column</div>";

        // Add verification_token_expires column
        $pdo->exec("ALTER TABLE users ADD COLUMN verification_token_expires DATETIME DEFAULT NULL");
        echo "<div class='status success'><strong>✓</strong> Added <code>verification_token_expires</code> column</div>";

        // Add index
        $pdo->exec("CREATE INDEX idx_verification_token ON users (verification_token)");
        echo "<div class='status success'><strong>✓</strong> Created index</div>";

        // Set OAuth users as verified
        $stmt = $pdo->exec("UPDATE users SET email_verified = TRUE WHERE oauth_provider IS NOT NULL");
        echo "<div class='status success'><strong>✓</strong> Set OAuth users as verified ($stmt users)</div>";

        echo "<div class='status success'>";
        echo "<h3>✅ Migration Completed Successfully!</h3>";
        echo "</div>";
    }

    // Show stats
    $stmt = $pdo->query("SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN email_verified = TRUE THEN 1 ELSE 0 END) as verified,
        SUM(CASE WHEN email_verified = FALSE THEN 1 ELSE 0 END) as unverified
        FROM users");
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);

    echo "<div class='status'>";
    echo "<h3>📊 User Statistics</h3>";
    echo "<p><strong>Total Users:</strong> " . $stats['total'] . "</p>";
    echo "<p><strong>Verified:</strong> " . $stats['verified'] . "</p>";
    echo "<p><strong>Unverified:</strong> " . $stats['unverified'] . "</p>";
    echo "</div>";

    echo "<div class='status warning'>";
    echo "<h3>⚠️ IMPORTANT - Security</h3>";
    echo "<p><strong>DELETE THIS FILE</strong> after successful migration!</p>";
    echo "</div>";

} catch (Exception $e) {
    echo "<div class='status error'>";
    echo "<h3>❌ Migration Failed</h3>";
    echo "<p><strong>Error:</strong> " . htmlspecialchars($e->getMessage()) . "</p>";
    echo "</div>";
}

echo "</body></html>";
?>

