<?php
require_once 'config.php';

$method = $_SERVER['REQUEST_METHOD'];
$pdo = getDBConnection();

switch ($method) {
    case 'GET':
        $action = $_GET['action'] ?? '';
        $userEmail = $_GET['email'] ?? '';
        
        switch ($action) {
            case 'usage':
                getTokenUsage($pdo, $userEmail);
                break;
            case 'balance':
                getTokenBalance($pdo, $userEmail);
                break;
            case 'settings':
                getUserSettings($pdo, $userEmail);
                break;
            default:
                http_response_code(400);
                echo json_encode(['error' => 'Invalid action']);
        }
        break;
        
    case 'POST':
        $data = json_decode(file_get_contents('php://input'), true);
        $action = $_GET['action'] ?? '';
        
        switch ($action) {
            case 'deduct':
                deductTokens($pdo, $data);
                break;
            case 'purchase':
                purchaseTokens($pdo, $data);
                break;
            case 'special-offer':
                handleSpecialOffer($pdo, $data);
                break;
            default:
                http_response_code(400);
                echo json_encode(['error' => 'Invalid action']);
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
}

function getTokenUsage($pdo, $userEmail) {
    if (!$userEmail) {
        http_response_code(400);
        echo json_encode(['error' => 'Email required']);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT feature, CAST(tokens_used AS UNSIGNED) as tokens_used, CAST(remaining_tokens AS UNSIGNED) as remaining_tokens, created_at 
            FROM token_usage 
            WHERE user_email = ? 
            ORDER BY created_at DESC 
            LIMIT 100
        ");
        $stmt->execute([$userEmail]);
        $usage = $stmt->fetchAll();
        
        echo json_encode([
            'success' => true,
            'usage' => $usage
        ]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to get token usage']);
    }
}

function getTokenBalance($pdo, $userEmail) {
    if (!$userEmail) {
        http_response_code(400);
        echo json_encode(['error' => 'Email required']);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("SELECT tokens FROM users WHERE email = ?");
        $stmt->execute([$userEmail]);
        $user = $stmt->fetch();
        
        if (!$user) {
            http_response_code(404);
            echo json_encode(['error' => 'User not found']);
            return;
        }
        
        echo json_encode([
            'success' => true,
            'tokens' => $user['tokens']
        ]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to get token balance']);
    }
}

function getUserSettings($pdo, $userEmail) {
    if (!$userEmail) {
        http_response_code(400);
        echo json_encode(['error' => 'Email required']);
        return;
    }
    
    try {
        $stmt = $pdo->prepare("SELECT special_offer_purchased, special_offer_shown FROM user_settings WHERE user_email = ?");
        $stmt->execute([$userEmail]);
        $settings = $stmt->fetch();
        
        if (!$settings) {
            // Create settings if they don't exist
            $stmt = $pdo->prepare("INSERT INTO user_settings (user_email) VALUES (?)");
            $stmt->execute([$userEmail]);
            $settings = ['special_offer_purchased' => false, 'special_offer_shown' => false];
        }
        
        echo json_encode([
            'success' => true,
            'settings' => $settings
        ]);
    } catch (PDOException $e) {
        http_response_code(500);
        echo json_encode(['error' => 'Failed to get user settings']);
    }
}

function cleanupOldTokenRecords($pdo, $userEmail) {
    try {
        // Delete records beyond the 100 most recent ones
        $stmt = $pdo->prepare("
            DELETE FROM token_usage 
            WHERE user_email = ? 
            AND id NOT IN (
                SELECT id FROM (
                    SELECT id 
                    FROM token_usage 
                    WHERE user_email = ? 
                    ORDER BY created_at DESC 
                    LIMIT 100
                ) AS recent_records
            )
        ");
        $stmt->execute([$userEmail, $userEmail]);
        
        return true;
    } catch (PDOException $e) {
        error_log("Cleanup error: " . $e->getMessage());
        return false;
    }
}

function deductTokens($pdo, $data) {
    if (!isset($data['email']) || !isset($data['amount']) || !isset($data['feature'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing required fields']);
        return;
    }
    
    $email = filter_var($data['email'], FILTER_SANITIZE_EMAIL);
    $amount = (int)$data['amount'];
    $feature = htmlspecialchars($data['feature'], ENT_QUOTES, 'UTF-8');
    
    try {
        $pdo->beginTransaction();
        
        // Get current token balance
        $stmt = $pdo->prepare("SELECT tokens FROM users WHERE email = ? FOR UPDATE");
        $stmt->execute([$email]);
        $user = $stmt->fetch();
        
        if (!$user) {
            $pdo->rollBack();
            http_response_code(404);
            echo json_encode(['error' => 'User not found']);
            return;
        }
        
        if ($user['tokens'] < $amount) {
            $pdo->rollBack();
            http_response_code(400);
            echo json_encode(['error' => 'Insufficient tokens']);
            return;
        }
        
        // Deduct tokens
        $newBalance = $user['tokens'] - $amount;
        $stmt = $pdo->prepare("UPDATE users SET tokens = ? WHERE email = ?");
        $stmt->execute([$newBalance, $email]);
        
        // Record usage
        $stmt = $pdo->prepare("
            INSERT INTO token_usage (user_email, feature, tokens_used, remaining_tokens) 
            VALUES (?, ?, ?, ?)
        ");
        $stmt->execute([$email, $feature, (int)$amount, (int)$newBalance]);
        
        // Clean up old records (keep only last 100)
        cleanupOldTokenRecords($pdo, $email);
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'remaining_tokens' => $newBalance,
            'tokens_used' => $amount
        ]);
    } catch (PDOException $e) {
        $pdo->rollBack();
        http_response_code(500);
        error_log("Token deduction error: " . $e->getMessage());
        echo json_encode([
            'error' => 'Failed to deduct tokens',
            'debug' => $e->getMessage()
        ]);
    }
}

function purchaseTokens($pdo, $data) {
    if (!isset($data['email']) || !isset($data['amount']) || !isset($data['tokens'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing required fields']);
        return;
    }
    
    $email = filter_var($data['email'], FILTER_SANITIZE_EMAIL);
    $amount = (float)$data['amount'];
    $tokens = (int)$data['tokens'];
    
    try {
        $pdo->beginTransaction();
        
        // Add tokens to user
        $stmt = $pdo->prepare("UPDATE users SET tokens = tokens + ? WHERE email = ?");
        $stmt->execute([$tokens, $email]);
        
        // Get updated balance
        $stmt = $pdo->prepare("SELECT tokens FROM users WHERE email = ?");
        $stmt->execute([$email]);
        $user = $stmt->fetch();
        
        // Clean up old records (keep only last 100)
        cleanupOldTokenRecords($pdo, $email);
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'tokens_added' => $tokens,
            'new_balance' => $user['tokens']
        ]);
    } catch (PDOException $e) {
        $pdo->rollBack();
        http_response_code(500);
        echo json_encode(['error' => 'Failed to purchase tokens']);
    }
}

function handleSpecialOffer($pdo, $data) {
    if (!isset($data['email']) || !isset($data['action'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing required fields']);
        return;
    }
    
    $email = filter_var($data['email'], FILTER_SANITIZE_EMAIL);
    $action = $data['action'];
    
    try {
        if ($action === 'purchase') {
            // Purchase 300 tokens for $10
            $pdo->beginTransaction();
            
            $stmt = $pdo->prepare("UPDATE users SET tokens = tokens + 300 WHERE email = ?");
            $stmt->execute([$email]);
            
            $stmt = $pdo->prepare("UPDATE user_settings SET special_offer_purchased = TRUE WHERE user_email = ?");
            $stmt->execute([$email]);
            
            // Clean up old records (keep only last 100)
            cleanupOldTokenRecords($pdo, $email);
            
            $pdo->commit();
            
            echo json_encode([
                'success' => true,
                'message' => 'Special offer purchased successfully',
                'tokens_added' => 300
            ]);
        } elseif ($action === 'mark-shown') {
            $stmt = $pdo->prepare("UPDATE user_settings SET special_offer_shown = TRUE WHERE user_email = ?");
            $stmt->execute([$email]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Special offer marked as shown'
            ]);
        } else {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid action']);
        }
    } catch (PDOException $e) {
        if (isset($pdo) && $pdo->inTransaction()) {
            $pdo->rollBack();
        }
        http_response_code(500);
        echo json_encode(['error' => 'Failed to handle special offer']);
    }
}
?> 