<?php
/**
 * Test Tool: Check User Tokens After Registration
 * This helps diagnose why users aren't getting tokens
 */

header('Content-Type: text/plain');
require_once 'config.php';

try {
    $conn = getDBConnection();
    
    echo "=== USER TOKENS DIAGNOSTIC ===\n\n";
    
    // Check last 10 registered users
    $stmt = $conn->query("SELECT id, email, name, tokens, created_at, oauth_provider FROM users ORDER BY id DESC LIMIT 10");
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo "Last 10 Registered Users:\n";
    echo str_repeat("=", 80) . "\n";
    
    foreach ($users as $user) {
        echo "\nUser ID: " . $user['id'] . "\n";
        echo "Email: " . $user['email'] . "\n";
        echo "Name: " . $user['name'] . "\n";
        echo "Tokens: " . $user['tokens'] . " ← ";
        
        if ($user['tokens'] == 0) {
            echo "❌ NO TOKENS!\n";
        } else {
            echo "✅ HAS TOKENS\n";
        }
        
        echo "Registration: " . ($user['oauth_provider'] ?? 'Email/Password') . "\n";
        echo "Created: " . $user['created_at'] . "\n";
        echo str_repeat("-", 80) . "\n";
    }
    
    // Count users by token status
    $stmt = $conn->query("SELECT 
        COUNT(*) as total_users,
        SUM(CASE WHEN tokens = 0 THEN 1 ELSE 0 END) as zero_tokens,
        SUM(CASE WHEN tokens > 0 THEN 1 ELSE 0 END) as has_tokens,
        AVG(tokens) as avg_tokens
    FROM users");
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    echo "\n\n=== STATISTICS ===\n";
    echo "Total Users: " . $stats['total_users'] . "\n";
    echo "Users with 0 tokens: " . $stats['zero_tokens'] . " ❌\n";
    echo "Users with tokens: " . $stats['has_tokens'] . " ✅\n";
    echo "Average tokens: " . round($stats['avg_tokens'], 2) . "\n";
    
    // Check table structure
    echo "\n\n=== USERS TABLE STRUCTURE ===\n";
    $stmt = $conn->query("DESCRIBE users");
    $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    foreach ($columns as $col) {
        echo $col['Field'] . " - Type: " . $col['Type'] . " - Default: " . ($col['Default'] ?? 'NULL') . "\n";
    }
    
    // Check if there's a trigger or constraint
    echo "\n\n=== CHECKING FOR TRIGGERS ===\n";
    $stmt = $conn->query("SHOW TRIGGERS LIKE 'users'");
    $triggers = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (count($triggers) > 0) {
        foreach ($triggers as $trigger) {
            echo "Trigger: " . $trigger['Trigger'] . "\n";
            echo "Event: " . $trigger['Event'] . "\n";
            echo "Timing: " . $trigger['Timing'] . "\n";
        }
    } else {
        echo "No triggers found\n";
    }
    
} catch (Exception $e) {
    echo "ERROR: " . $e->getMessage();
}
?>

