<?php
/**
 * Installation verification script
 * This will check if all required files are present and working
 */

header('Content-Type: application/json');

$checks = [];
$errors = [];

// Check 1: Database connection
try {
    require_once __DIR__ . '/../config.php';
    $pdo = getDBConnection();
    $checks['database'] = 'Connected successfully';
} catch (Exception $e) {
    $errors['database'] = 'Database connection failed: ' . $e->getMessage();
}

// Check 2: Stripe library
try {
    require_once 'vendor/autoload.php';
    if (class_exists('Stripe\Stripe')) {
        $checks['stripe_library'] = 'Loaded successfully (v' . \Stripe\Stripe::VERSION . ')';
    } else {
        $errors['stripe_library'] = 'Stripe class not found';
    }
} catch (Exception $e) {
    $errors['stripe_library'] = 'Stripe library error: ' . $e->getMessage();
}

// Check 3: Stripe configuration
try {
    require_once __DIR__ . '/config.php';
    if (defined('STRIPE_SECRET_KEY') && !empty(STRIPE_SECRET_KEY)) {
        $checks['stripe_config'] = 'Secret key configured';
    } else {
        $errors['stripe_config'] = 'Stripe secret key not configured';
    }
} catch (Exception $e) {
    $errors['stripe_config'] = 'Stripe config error: ' . $e->getMessage();
}

// Check 4: Required files
$requiredFiles = [
    'config.php' => 'Stripe configuration',
    'create-payment-intent.php' => 'Payment intent creation',
    'confirm-payment.php' => 'Payment confirmation',
    'vendor/autoload.php' => 'Stripe autoloader',
    'vendor/stripe/stripe-php/init.php' => 'Stripe library init'
];

foreach ($requiredFiles as $file => $description) {
    if (file_exists($file)) {
        $checks['file_' . $file] = $description . ' - Present';
    } else {
        $errors['file_' . $file] = $description . ' - Missing';
    }
}

// Check 5: File permissions
$filesToCheck = [
    'config.php' => 'Stripe config',
    'create-payment-intent.php' => 'Payment intent',
    'confirm-payment.php' => 'Payment confirmation'
];

foreach ($filesToCheck as $file => $description) {
    if (file_exists($file)) {
        $perms = fileperms($file);
        $perms = substr(sprintf('%o', $perms), -4);
        if ($perms == '0644' || $perms == '0640') {
            $checks['perms_' . $file] = $description . ' - Good permissions (' . $perms . ')';
        } else {
            $errors['perms_' . $file] = $description . ' - Check permissions (' . $perms . ')';
        }
    }
}

// Prepare response
$response = [
    'success' => empty($errors),
    'timestamp' => date('Y-m-d H:i:s'),
    'checks' => $checks,
    'errors' => $errors,
    'summary' => [
        'total_checks' => count($checks) + count($errors),
        'passed' => count($checks),
        'failed' => count($errors)
    ]
];

if (empty($errors)) {
    $response['message'] = '✅ All checks passed! Your Stripe installation is ready.';
} else {
    $response['message'] = '❌ Some checks failed. Please review the errors below.';
}

echo json_encode($response, JSON_PRETTY_PRINT);
?> 