<?php
// Set headers first to prevent any output issues
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

try {
    // Check if config file exists
    if (!file_exists('config.php')) {
        throw new Exception('Configuration file not found');
    }
    
    require_once 'config.php';
    
    // Check if Stripe is loaded
    if (!class_exists('Stripe\Stripe')) {
        throw new Exception('Stripe library not loaded');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['amount']) || !isset($input['tokens']) || !isset($input['email'])) {
        http_response_code(400);
        echo json_encode(['error' => 'Missing required fields']);
        exit;
    }
    
    $amount = (int)$input['amount'];
    $tokens = (int)$input['tokens'];
    $email = filter_var($input['email'], FILTER_SANITIZE_EMAIL);
    
    // Create payment intent with PaymentElement support
    $paymentIntent = \Stripe\PaymentIntent::create([
        'amount' => $amount,
        'currency' => 'usd',
        'metadata' => [
            'tokens' => $tokens,
            'email' => $email
        ],
        'automatic_payment_methods' => [
            'enabled' => true,
        ],
    ]);
    
    echo json_encode([
        'success' => true,
        'clientSecret' => $paymentIntent->client_secret,
        'paymentIntentId' => $paymentIntent->id
    ]);
    
} catch (\Stripe\Exception\ApiErrorException $e) {
    http_response_code(400);
    echo json_encode(['error' => 'Stripe API Error: ' . $e->getMessage()]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server Error: ' . $e->getMessage()]);
}
?> 