-- Drop the trigger if it exists
DROP TRIGGER IF EXISTS cleanup_token_usage_trigger;

-- Create a trigger to automatically clean up old token usage records
DELIMITER $$

CREATE TRIGGER cleanup_token_usage_trigger
AFTER INSERT ON token_usage
FOR EACH ROW
BEGIN
    DECLARE record_count INT;
    DECLARE oldest_id INT;
    
    -- Get the count of records for this user
    SELECT COUNT(*) INTO record_count 
    FROM token_usage 
    WHERE user_email = NEW.user_email;
    
    -- If we have more than 100 records, delete the oldest ones
    IF record_count > 100 THEN
        -- Get the ID of the oldest record to delete
        SELECT id INTO oldest_id
        FROM token_usage 
        WHERE user_email = NEW.user_email 
        ORDER BY created_at ASC 
        LIMIT 1;
        
        -- Delete the oldest record
        DELETE FROM token_usage 
        WHERE id = oldest_id;
    END IF;
END$$

DELIMITER ;

-- Alternative approach: Create a stored procedure for manual cleanup
DROP PROCEDURE IF EXISTS cleanup_token_usage;

DELIMITER $$

CREATE PROCEDURE cleanup_token_usage()
BEGIN
    DECLARE done INT DEFAULT FALSE;
    DECLARE user_email VARCHAR(255);
    DECLARE record_count INT;
    DECLARE user_cursor CURSOR FOR 
        SELECT DISTINCT user_email FROM token_usage;
    DECLARE CONTINUE HANDLER FOR NOT FOUND SET done = TRUE;
    
    OPEN user_cursor;
    
    read_loop: LOOP
        FETCH user_cursor INTO user_email;
        IF done THEN
            LEAVE read_loop;
        END IF;
        
        -- Get count of records for this user
        SELECT COUNT(*) INTO record_count 
        FROM token_usage 
        WHERE user_email = user_email;
        
        -- If more than 100 records, delete oldest ones
        IF record_count > 100 THEN
            DELETE FROM token_usage 
            WHERE user_email = user_email 
            AND id NOT IN (
                SELECT id FROM (
                    SELECT id 
                    FROM token_usage 
                    WHERE user_email = user_email 
                    ORDER BY created_at DESC 
                    LIMIT 100
                ) AS keep_records
            );
        END IF;
    END LOOP;
    
    CLOSE user_cursor;
END$$

DELIMITER ;

-- Create an event to run cleanup automatically (optional)
-- This will run the cleanup procedure every day at 2 AM
-- Uncomment the following lines if you want automatic daily cleanup

/*
DROP EVENT IF EXISTS daily_token_usage_cleanup;

CREATE EVENT daily_token_usage_cleanup
ON SCHEDULE EVERY 1 DAY
STARTS CURRENT_TIMESTAMP
DO
BEGIN
    CALL cleanup_token_usage();
END;
*/

-- Show the created trigger
SHOW TRIGGERS LIKE 'token_usage';

-- Show the created procedure
SHOW PROCEDURE STATUS LIKE 'cleanup_token_usage'; 