<?php
/**
 * Send Email Verification
 * Sends a verification email to the user's email address
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

// Use local config for localhost, production config otherwise
if ($_SERVER['HTTP_HOST'] === 'localhost' || $_SERVER['HTTP_HOST'] === '127.0.0.1' || strpos($_SERVER['HTTP_HOST'], 'localhost:') === 0) {
    require_once 'config_local.php';
} else {
    require_once 'config.php';
}

try {
    $pdo = getDBConnection();
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Database connection failed']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);
$email = $input['email'] ?? '';

if (empty($email)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Email is required']);
    exit;
}

// Check if user exists
$stmt = $pdo->prepare("SELECT id, name, email, email_verified FROM users WHERE email = ?");
$stmt->execute([$email]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$user) {
    http_response_code(404);
    echo json_encode(['success' => false, 'message' => 'User not found']);
    exit;
}

// Check if already verified
if ($user['email_verified']) {
    echo json_encode([
        'success' => false, 
        'message' => 'Email is already verified',
        'already_verified' => true
    ]);
    exit;
}

// Generate verification token
$verificationToken = bin2hex(random_bytes(32));
$tokenExpires = date('Y-m-d H:i:s', strtotime('+24 hours'));

// Update user with verification token
$stmt = $pdo->prepare("UPDATE users SET verification_token = ?, verification_token_expires = ? WHERE email = ?");
$stmt->execute([$verificationToken, $tokenExpires, $email]);

// Determine the verification URL
$isLocalhost = $_SERVER['HTTP_HOST'] === 'localhost' || strpos($_SERVER['HTTP_HOST'], 'localhost') !== false;
$baseUrl = $isLocalhost ? 'http://localhost:5003' : 'https://app.tofeliq.com';
$verificationUrl = $baseUrl . '/?verify-email=' . $verificationToken;

// Email content
$subject = "Verify Your Email - TOEFL Analyzer";
$userName = $user['name'] ?? 'User';

$emailBody = "
<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <style>
        body { font-family: Arial, sans-serif; background-color: #f4f7fa; margin: 0; padding: 0; }
        .email-container { max-width: 600px; margin: 40px auto; background: #ffffff; border-radius: 16px; overflow: hidden; box-shadow: 0 4px 20px rgba(0,0,0,0.1); }
        .email-header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px 30px; text-align: center; }
        .email-header h1 { color: #ffffff; margin: 0; font-size: 28px; }
        .email-body { padding: 40px 30px; }
        .verify-button { display: inline-block; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: #ffffff !important; text-decoration: none; padding: 16px 40px; border-radius: 8px; font-weight: 600; }
        .button-container { text-align: center; margin: 30px 0; }
    </style>
</head>
<body>
    <div class='email-container'>
        <div class='email-header'>
            <h1>🎯 TOEFL Analyzer</h1>
            <p style='color: rgba(255,255,255,0.9); margin: 10px 0 0;'>Email Verification</p>
        </div>
        <div class='email-body'>
            <h2 style='color: #2c3e50;'>Hello, $userName! 👋</h2>
            <p>Thank you for registering with TOEFL Analyzer.</p>
            <p>To verify your email address, click the button below:</p>
            <div class='button-container'>
                <a href='$verificationUrl' class='verify-button'>✅ Verify My Email</a>
            </div>
            <p style='background: #fff3cd; padding: 15px; border-radius: 4px; border-left: 4px solid #f39c12; color: #856404;'>
                <strong>⏰ Important:</strong> This link expires in 24 hours.
            </p>
            <p>If the button doesn't work, copy this link: <br><a href='$verificationUrl'>$verificationUrl</a></p>
        </div>
    </div>
</body>
</html>
";

$headers = "MIME-Version: 1.0" . "\r\n";
$headers .= "Content-type: text/html; charset=UTF-8" . "\r\n";
$headers .= "From: TOEFL Analyzer <noreply@tofeliq.com>" . "\r\n";

$emailSent = mail($email, $subject, $emailBody, $headers);

if ($emailSent) {
    echo json_encode([
        'success' => true,
        'message' => 'Verification email sent! Please check your inbox.'
    ]);
} else {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Failed to send verification email.'
    ]);
}
?>

