<?php
/**
 * OpenAI API Backend Handler
 * This securely handles all OpenAI API requests from the frontend
 * The API key is stored server-side and never exposed to clients
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once 'config_openai.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed. Use POST.']);
    exit;
}

// Check if API key is configured
if (!isApiKeyConfigured()) {
    http_response_code(500);
    echo json_encode([
        'error' => 'OpenAI API key not configured',
        'message' => 'Please configure your OpenAI API key in api/config_openai.php',
        'useMockData' => true
    ]);
    exit;
}

// Get request body
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['prompt'])) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing required field: prompt']);
    exit;
}

$prompt = $input['prompt'];
$systemPrompt = $input['systemPrompt'] ?? 'You are an expert TOEFL tutor and academic writing specialist. Provide detailed, accurate, and practical analysis for TOEFL students. Focus on academic writing skills, integrated tasks, university-level vocabulary, and actionable insights for TOEFL exam success.';
$maxTokens = $input['maxTokens'] ?? OPENAI_MAX_TOKENS;
$temperature = $input['temperature'] ?? OPENAI_TEMPERATURE;
$model = $input['model'] ?? OPENAI_MODEL;

// Check cache if enabled
$cacheKey = md5($prompt . $systemPrompt . $model);
$cacheFile = __DIR__ . '/cache/openai_' . $cacheKey . '.json';

if (OPENAI_CACHE_ENABLED && file_exists($cacheFile)) {
    $cacheAge = time() - filemtime($cacheFile);
    if ($cacheAge < OPENAI_CACHE_DURATION) {
        // Return cached response
        $cachedData = json_decode(file_get_contents($cacheFile), true);
        echo json_encode([
            'success' => true,
            'content' => $cachedData['content'],
            'cached' => true,
            'cacheAge' => $cacheAge
        ]);
        exit;
    }
}

// Prepare OpenAI API request
$apiKey = getOpenAIApiKey();
$requestData = [
    'model' => $model,
    'messages' => [
        [
            'role' => 'system',
            'content' => $systemPrompt
        ],
        [
            'role' => 'user',
            'content' => $prompt
        ]
    ],
    'max_tokens' => intval($maxTokens),
    'temperature' => floatval($temperature)
];

// Initialize cURL
$ch = curl_init(OPENAI_API_URL);
curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
curl_setopt($ch, CURLOPT_POST, true);
curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($requestData));
curl_setopt($ch, CURLOPT_HTTPHEADER, [
    'Content-Type: application/json',
    'Authorization: Bearer ' . $apiKey
]);
curl_setopt($ch, CURLOPT_TIMEOUT, 30);

// Execute request
$response = curl_exec($ch);
$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
$curlError = curl_error($ch);
curl_close($ch);

// Handle cURL errors
if ($curlError) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Request failed',
        'message' => $curlError,
        'useMockData' => true
    ]);
    exit;
}

// Parse OpenAI response
$responseData = json_decode($response, true);

// Handle API errors
if ($httpCode !== 200) {
    $errorMessage = 'OpenAI API error';
    $useMockData = false;
    
    if (isset($responseData['error'])) {
        $errorMessage = $responseData['error']['message'] ?? $errorMessage;
        $errorType = $responseData['error']['type'] ?? '';
        
        // Check for quota/billing errors
        if (in_array($errorType, ['insufficient_quota', 'billing_not_active', 'invalid_api_key'])) {
            $useMockData = true;
        }
    }
    
    http_response_code($httpCode);
    echo json_encode([
        'error' => $errorMessage,
        'httpCode' => $httpCode,
        'useMockData' => $useMockData,
        'details' => $responseData
    ]);
    exit;
}

// Extract content from response
if (!isset($responseData['choices'][0]['message']['content'])) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Invalid response from OpenAI',
        'useMockData' => true
    ]);
    exit;
}

$content = $responseData['choices'][0]['message']['content'];

// Cache the successful response
if (OPENAI_CACHE_ENABLED) {
    // Create cache directory if it doesn't exist
    $cacheDir = __DIR__ . '/cache';
    if (!is_dir($cacheDir)) {
        mkdir($cacheDir, 0755, true);
    }
    
    file_put_contents($cacheFile, json_encode([
        'content' => $content,
        'timestamp' => time(),
        'prompt' => $prompt
    ]));
}

// Return successful response
echo json_encode([
    'success' => true,
    'content' => $content,
    'cached' => false,
    'usage' => $responseData['usage'] ?? null
]);
?>

