<?php
require_once 'config.php';
require_once 'oauth-config.php';

$method = $_SERVER['REQUEST_METHOD'];
$pdo = getDBConnection();

switch ($method) {
    case 'GET':
        $action = $_GET['action'] ?? '';
        
        switch ($action) {
            case 'google-auth':
                handleGoogleAuth();
                break;
            case 'facebook-auth':
                handleFacebookAuth();
                break;
            case 'google-callback':
                handleGoogleCallback($pdo);
                break;
            case 'facebook-callback':
                handleFacebookCallback($pdo);
                break;
            default:
                http_response_code(400);
                echo json_encode(['error' => 'Invalid action']);
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
}

function handleGoogleAuth() {
    $state = generateOAuthState();
    
    // Store state in session for verification
    session_start();
    $_SESSION['oauth_state'] = $state;
    
    $authUrl = getGoogleAuthUrl($state);
    
    header('Location: ' . $authUrl);
    exit;
}

function handleFacebookAuth() {
    $state = generateOAuthState();
    
    // Store state in session for verification
    session_start();
    $_SESSION['oauth_state'] = $state;
    
    $authUrl = getFacebookAuthUrl($state);
    
    header('Location: ' . $authUrl);
    exit;
}

function handleGoogleCallback($pdo) {
    session_start();
    
    $code = $_GET['code'] ?? '';
    $state = $_GET['state'] ?? '';
    $storedState = $_SESSION['oauth_state'] ?? '';
    
    if (!$code || !$state || !verifyOAuthState($state, $storedState)) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid OAuth callback']);
        return;
    }
    
    try {
        // Exchange code for access token
        $tokenData = exchangeGoogleCodeForToken($code);
        
        if (!isset($tokenData['access_token'])) {
            throw new Exception('Failed to get access token');
        }
        
        // Get user info from Google
        $userInfo = getGoogleUserInfo($tokenData['access_token']);
        
        if (!isset($userInfo['email'])) {
            throw new Exception('Failed to get user email from Google');
        }
        
        // Handle user login/registration
        $result = handleOAuthUser($pdo, 'google', $userInfo['id'], $userInfo['email'], $userInfo['name']);
        
        // Clear OAuth state
        unset($_SESSION['oauth_state']);
        
        // Redirect to frontend with result
        $redirectUrl = 'https://app.tofeliq.com/?' . http_build_query($result);
        header('Location: ' . $redirectUrl);
        exit;
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'OAuth callback failed: ' . $e->getMessage()]);
    }
}

function handleFacebookCallback($pdo) {
    session_start();
    
    $code = $_GET['code'] ?? '';
    $state = $_GET['state'] ?? '';
    $storedState = $_SESSION['oauth_state'] ?? '';
    
    if (!$code || !$state || !verifyOAuthState($state, $storedState)) {
        http_response_code(400);
        echo json_encode(['error' => 'Invalid OAuth callback']);
        return;
    }
    
    try {
        // Exchange code for access token
        $tokenData = exchangeFacebookCodeForToken($code);
        
        if (!isset($tokenData['access_token'])) {
            throw new Exception('Failed to get access token');
        }
        
        // Get user info from Facebook
        $userInfo = getFacebookUserInfo($tokenData['access_token']);
        
        if (!isset($userInfo['email'])) {
            throw new Exception('Failed to get user email from Facebook');
        }
        
        // Handle user login/registration
        $result = handleOAuthUser($pdo, 'facebook', $userInfo['id'], $userInfo['email'], $userInfo['name']);
        
        // Clear OAuth state
        unset($_SESSION['oauth_state']);
        
        // Redirect to frontend with result
        $redirectUrl = 'https://app.tofeliq.com/?' . http_build_query($result);
        header('Location: ' . $redirectUrl);
        exit;
        
    } catch (Exception $e) {
        http_response_code(500);
        echo json_encode(['error' => 'OAuth callback failed: ' . $e->getMessage()]);
    }
}

function handleOAuthUser($pdo, $provider, $oauthId, $email, $name) {
    try {
        // Check if user exists by OAuth ID
        $stmt = $pdo->prepare("SELECT id, email, name, tokens, email_verified FROM users WHERE oauth_provider = ? AND oauth_id = ?");
        $stmt->execute([$provider, $oauthId]);
        $user = $stmt->fetch();
        
        if ($user) {
            // User exists, return login success
            return [
                'success' => 'true',
                'action' => 'login',
                'user' => json_encode([
                    'id' => $user['id'],
                    'email' => $user['email'],
                    'name' => $user['name'],
                    'tokens' => $user['tokens'],
                    'oauth_provider' => $provider,
                    'isEmailVerified' => true, // OAuth users are always verified
                    'email_verified' => true
                ])
            ];
        }
        
        // Check if user exists by email
        $stmt = $pdo->prepare("SELECT id, email, name, tokens, email_verified FROM users WHERE email = ?");
        $stmt->execute([$email]);
        $user = $stmt->fetch();
        
        if ($user) {
            // User exists with email but no OAuth, link the accounts and set as verified
            $stmt = $pdo->prepare("UPDATE users SET oauth_provider = ?, oauth_id = ?, email_verified = TRUE WHERE email = ?");
            $stmt->execute([$provider, $oauthId, $email]);
            
            return [
                'success' => 'true',
                'action' => 'login',
                'user' => json_encode([
                    'id' => $user['id'],
                    'email' => $user['email'],
                    'name' => $user['name'],
                    'tokens' => $user['tokens'],
                    'oauth_provider' => $provider,
                    'isEmailVerified' => true, // OAuth users are always verified
                    'email_verified' => true
                ])
            ];
        }
        
        // Create new user (OAuth users are automatically verified, no free tokens)
        $pdo->beginTransaction();
        
        $stmt = $pdo->prepare("INSERT INTO users (email, name, oauth_provider, oauth_id, tokens, email_verified) VALUES (?, ?, ?, ?, 0, TRUE)");
        $stmt->execute([$email, $name, $provider, $oauthId]);
        
        $userId = $pdo->lastInsertId();
        
        // Create user settings
        $stmt = $pdo->prepare("INSERT INTO user_settings (user_email) VALUES (?)");
        $stmt->execute([$email]);
        
        $pdo->commit();
        
        return [
            'success' => 'true',
            'action' => 'register',
            'user' => json_encode([
                'id' => $userId,
                'email' => $email,
                'name' => $name,
                'tokens' => 0,
                'oauth_provider' => $provider,
                'isEmailVerified' => true, // OAuth users are always verified
                'email_verified' => true
            ])
        ];
        
    } catch (PDOException $e) {
        $pdo->rollBack();
        return [
            'success' => 'false',
            'error' => 'Database error: ' . $e->getMessage()
        ];
    }
}
?> 