<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Include the production config
require_once 'config.php';

// Error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Log function for debugging
function logError($message) {
    error_log("TOEFL API Error: " . $message);
}

try {
    // Create database connection with better error handling
    $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
    
    if ($conn->connect_error) {
        logError("Database connection failed: " . $conn->connect_error);
        echo json_encode([
            'success' => false, 
            'message' => 'Database connection failed',
            'error' => 'DB_CONNECTION_ERROR'
        ]);
        exit();
    }
    
    // Set charset
    $conn->set_charset("utf8mb4");
    
} catch (Exception $e) {
    logError("Database exception: " . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'Database connection failed',
        'error' => 'DB_EXCEPTION'
    ]);
    exit();
}

// Get input data
$input = null;
$action = '';

try {
    // Try to get JSON input
    $rawInput = file_get_contents('php://input');
    if (!empty($rawInput)) {
        $input = json_decode($rawInput, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Invalid JSON input');
        }
    }
    
    // Get action from POST data or URL parameter
    $action = $input['action'] ?? $_GET['action'] ?? $_POST['action'] ?? '';
    
    if (empty($action)) {
        echo json_encode([
            'success' => false, 
            'message' => 'No action specified',
            'error' => 'NO_ACTION'
        ]);
        exit();
    }
    
} catch (Exception $e) {
    logError("Input parsing error: " . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'Invalid input data',
        'error' => 'INPUT_ERROR'
    ]);
    exit();
}

// Route to appropriate handler
try {
    switch ($action) {
        case 'login':
            handleLogin($conn, $input);
            break;
        case 'register':
            handleRegister($conn, $input);
            break;
        case 'reset_password':
            handlePasswordReset($conn, $input);
            break;
        case 'reset_confirm':
            handlePasswordResetConfirm($conn, $input);
            break;
        case 'update_profile':
            handleProfileUpdate($conn, $input);
            break;
        case 'update_password':
            handlePasswordUpdate($conn, $input);
            break;
        default:
            echo json_encode([
                'success' => false, 
                'message' => 'Invalid action: ' . $action,
                'error' => 'INVALID_ACTION'
            ]);
    }
} catch (Exception $e) {
    logError("Handler error: " . $e->getMessage());
    echo json_encode([
        'success' => false, 
        'message' => 'Internal server error',
        'error' => 'HANDLER_ERROR'
    ]);
}

$conn->close();

function handleLogin($conn, $input) {
    $email = trim($input['email'] ?? '');
    $password = $input['password'] ?? '';

    if (empty($email) || empty($password)) {
        echo json_encode([
            'success' => false, 
            'message' => 'Email and password are required',
            'error' => 'MISSING_CREDENTIALS'
        ]);
        return;
    }

    try {
        $stmt = $conn->prepare("SELECT id, email, password, name, tokens FROM users WHERE email = ?");
        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }
        
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 1) {
            $user = $result->fetch_assoc();
            if (password_verify($password, $user['password'])) {
                // Remove password from response
                unset($user['password']);
                // Add email verification status
                $user['isEmailVerified'] = isset($user['is_email_verified']) ? (bool)$user['is_email_verified'] : false;
                
                echo json_encode([
                    'success' => true, 
                    'user' => $user,
                    'message' => 'Login successful'
                ]);
            } else {
                echo json_encode([
                    'success' => false, 
                    'message' => 'Invalid password',
                    'error' => 'INVALID_PASSWORD'
                ]);
            }
        } else {
            echo json_encode([
                'success' => false, 
                'message' => 'User not found',
                'error' => 'USER_NOT_FOUND'
            ]);
        }
        $stmt->close();
        
    } catch (Exception $e) {
        logError("Login error: " . $e->getMessage());
        echo json_encode([
            'success' => false, 
            'message' => 'Login failed',
            'error' => 'LOGIN_ERROR'
        ]);
    }
}

function handleRegister($conn, $input) {
    $email = trim($input['email'] ?? '');
    $password = $input['password'] ?? '';
    
    // Automatically extract username from email
    $name = trim($input['name'] ?? '');
    if (empty($name) && !empty($email)) {
        // Extract username from email (part before @)
        $name = explode('@', $email)[0];
        // Capitalize first letter
        $name = ucfirst(strtolower($name));
    }

    if (empty($email) || empty($password)) {
        echo json_encode([
            'success' => false, 
            'message' => 'Email and password are required',
            'error' => 'MISSING_CREDENTIALS'
        ]);
        return;
    }

    // Validate email format
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode([
            'success' => false, 
            'message' => 'Invalid email format',
            'error' => 'INVALID_EMAIL'
        ]);
        return;
    }

    // Validate password strength
    if (strlen($password) < 6) {
        echo json_encode([
            'success' => false, 
            'message' => 'Password must be at least 6 characters',
            'error' => 'WEAK_PASSWORD'
        ]);
        return;
    }

    try {
        // Check if user already exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }
        
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows > 0) {
            echo json_encode([
                'success' => false, 
                'message' => 'User already exists',
                'error' => 'USER_EXISTS'
            ]);
            $stmt->close();
            return;
        }
        $stmt->close();

        // Hash password
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        
        // Start transaction
        $conn->begin_transaction();
        
        try {
            // Insert user
            $stmt = $conn->prepare("INSERT INTO users (email, password, name, tokens, created_at) VALUES (?, ?, ?, 50, NOW())");
            if (!$stmt) {
                throw new Exception('Prepare failed: ' . $conn->error);
            }
            
            $stmt->bind_param("sss", $email, $hashedPassword, $name);
            $stmt->execute();
            $userId = $conn->insert_id;
            $stmt->close();

            // Create user settings if table exists
            try {
                $stmt = $conn->prepare("INSERT INTO user_settings (user_id, theme, notifications, language) VALUES (?, 'light', 1, 'en')");
                if ($stmt) {
                    $stmt->bind_param("i", $userId);
                    $stmt->execute();
                    $stmt->close();
                }
            } catch (Exception $e) {
                // User settings table might not exist, continue
                logError("User settings creation failed: " . $e->getMessage());
            }

            $conn->commit();

            // Get user data for response
            $stmt = $conn->prepare("SELECT id, email, name, tokens FROM users WHERE id = ?");
            if (!$stmt) {
                throw new Exception('Prepare failed: ' . $conn->error);
            }
            
            $stmt->bind_param("i", $userId);
            $stmt->execute();
            $result = $stmt->get_result();
            $user = $result->fetch_assoc();
            $stmt->close();

            // Add email verification status
            $user['isEmailVerified'] = false;

            echo json_encode([
                'success' => true, 
                'user' => $user,
                'message' => 'Registration successful'
            ]);
            
        } catch (Exception $e) {
            $conn->rollback();
            throw $e;
        }
        
    } catch (Exception $e) {
        logError("Registration error: " . $e->getMessage());
        echo json_encode([
            'success' => false, 
            'message' => 'Registration failed',
            'error' => 'REGISTRATION_ERROR'
        ]);
    }
}

function handlePasswordReset($conn, $input) {
    $email = trim($input['email'] ?? '');

    if (empty($email)) {
        echo json_encode([
            'success' => false, 
            'message' => 'Email is required',
            'error' => 'MISSING_EMAIL'
        ]);
        return;
    }

    try {
        // Check if user exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }
        
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 0) {
            echo json_encode([
                'success' => false, 
                'message' => 'User not found',
                'error' => 'USER_NOT_FOUND'
            ]);
            $stmt->close();
            return;
        }
        $stmt->close();

        // Generate reset token
        $token = bin2hex(random_bytes(32));
        $expires = date('Y-m-d H:i:s', strtotime('+1 hour'));

        // Store reset token (create table if it doesn't exist)
        try {
            $stmt = $conn->prepare("INSERT INTO password_resets (email, token, expires_at) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE token = ?, expires_at = ?");
            if (!$stmt) {
                // Try to create the table first
                $conn->query("CREATE TABLE IF NOT EXISTS password_resets (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    email VARCHAR(255) NOT NULL,
                    token VARCHAR(255) NOT NULL,
                    expires_at TIMESTAMP NOT NULL,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    used TINYINT(1) DEFAULT 0,
                    KEY email_idx (email),
                    KEY token_idx (token)
                )");
                
                $stmt = $conn->prepare("INSERT INTO password_resets (email, token, expires_at) VALUES (?, ?, ?)");
                $stmt->bind_param("sss", $email, $token, $expires);
            } else {
                $stmt->bind_param("sssss", $email, $token, $expires, $token, $expires);
            }
            
            $stmt->execute();
            $stmt->close();

            // In a real application, you would send an email here
            // For now, we'll return the reset link for testing purposes
            $resetLink = "https://app.tofeliq.com/?token=" . $token;
            echo json_encode([
                'success' => true, 
                'message' => 'Password reset link sent to your email',
                'resetLink' => $resetLink // Include reset link for testing
            ]);
            
        } catch (Exception $e) {
            logError("Password reset error: " . $e->getMessage());
            echo json_encode([
                'success' => false, 
                'message' => 'Password reset failed',
                'error' => 'RESET_ERROR'
            ]);
        }
        
    } catch (Exception $e) {
        logError("Password reset error: " . $e->getMessage());
        echo json_encode([
            'success' => false, 
            'message' => 'Password reset failed',
            'error' => 'RESET_ERROR'
        ]);
    }
}

function handlePasswordResetConfirm($conn, $input) {
    $token = $input['token'] ?? '';
    $password = $input['password'] ?? '';

    if (empty($token) || empty($password)) {
        echo json_encode([
            'success' => false, 
            'message' => 'Token and password are required',
            'error' => 'MISSING_DATA'
        ]);
        return;
    }

    try {
        // Check if token is valid and not expired
        $stmt = $conn->prepare("SELECT email FROM password_resets WHERE token = ? AND expires_at > NOW() AND used = 0");
        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }
        
        $stmt->bind_param("s", $token);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 0) {
            echo json_encode([
                'success' => false, 
                'message' => 'Invalid or expired token',
                'error' => 'INVALID_TOKEN'
            ]);
            $stmt->close();
            return;
        }

        $reset = $result->fetch_assoc();
        $email = $reset['email'];
        $stmt->close();

        // Update password
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $conn->prepare("UPDATE users SET password = ? WHERE email = ?");
        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }
        
        $stmt->bind_param("ss", $hashedPassword, $email);
        $stmt->execute();
        $stmt->close();

        // Mark token as used
        $stmt = $conn->prepare("UPDATE password_resets SET used = 1 WHERE token = ?");
        if ($stmt) {
            $stmt->bind_param("s", $token);
            $stmt->execute();
            $stmt->close();
        }

        echo json_encode([
            'success' => true, 
            'message' => 'Password updated successfully'
        ]);
        
    } catch (Exception $e) {
        logError("Password reset confirm error: " . $e->getMessage());
        echo json_encode([
            'success' => false, 
            'message' => 'Password update failed',
            'error' => 'UPDATE_ERROR'
        ]);
    }
}

function handleProfileUpdate($conn, $input) {
    $email = trim($input['email'] ?? '');
    $name = trim($input['name'] ?? '');
    $newEmail = trim($input['newEmail'] ?? '');

    if (empty($email)) {
        echo json_encode([
            'success' => false, 
            'message' => 'Current email is required',
            'error' => 'MISSING_EMAIL'
        ]);
        return;
    }

    if (empty($name)) {
        echo json_encode([
            'success' => false, 
            'message' => 'Name is required',
            'error' => 'MISSING_NAME'
        ]);
        return;
    }

    try {
        // Check if user exists
        $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }
        
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 0) {
            echo json_encode([
                'success' => false, 
                'message' => 'User not found',
                'error' => 'USER_NOT_FOUND'
            ]);
            $stmt->close();
            return;
        }
        $stmt->close();

        // If email is being changed, check if new email already exists
        if (!empty($newEmail) && $newEmail !== $email) {
            if (!filter_var($newEmail, FILTER_VALIDATE_EMAIL)) {
                echo json_encode([
                    'success' => false, 
                    'message' => 'Invalid email format',
                    'error' => 'INVALID_EMAIL'
                ]);
                return;
            }
            
            $stmt = $conn->prepare("SELECT id FROM users WHERE email = ?");
            if (!$stmt) {
                throw new Exception('Prepare failed: ' . $conn->error);
            }
            
            $stmt->bind_param("s", $newEmail);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result->num_rows > 0) {
                echo json_encode([
                    'success' => false, 
                    'message' => 'Email already exists',
                    'error' => 'EMAIL_EXISTS'
                ]);
                $stmt->close();
                return;
            }
            $stmt->close();
        }

        // Update user profile
        $updateEmail = !empty($newEmail) ? $newEmail : $email;
        $stmt = $conn->prepare("UPDATE users SET name = ?, email = ? WHERE email = ?");
        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }
        
        $stmt->bind_param("sss", $name, $updateEmail, $email);
        $stmt->execute();
        $stmt->close();

        // Get updated user data
        $stmt = $conn->prepare("SELECT id, email, name, tokens FROM users WHERE email = ?");
        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }
        
        $stmt->bind_param("s", $updateEmail);
        $stmt->execute();
        $result = $stmt->get_result();
        $user = $result->fetch_assoc();
        $stmt->close();

        echo json_encode([
            'success' => true, 
            'user' => $user, 
            'message' => 'Profile updated successfully'
        ]);
        
    } catch (Exception $e) {
        logError("Profile update error: " . $e->getMessage());
        echo json_encode([
            'success' => false, 
            'message' => 'Profile update failed',
            'error' => 'UPDATE_ERROR'
        ]);
    }
}

function handlePasswordUpdate($conn, $input) {
    $email = trim($input['email'] ?? '');
    $currentPassword = $input['currentPassword'] ?? '';
    $newPassword = $input['newPassword'] ?? '';

    if (empty($email) || empty($currentPassword) || empty($newPassword)) {
        echo json_encode([
            'success' => false, 
            'message' => 'Email, current password, and new password are required',
            'error' => 'MISSING_DATA'
        ]);
        return;
    }

    if (strlen($newPassword) < 6) {
        echo json_encode([
            'success' => false, 
            'message' => 'New password must be at least 6 characters',
            'error' => 'WEAK_PASSWORD'
        ]);
        return;
    }

    try {
        // Check if user exists and verify current password
        $stmt = $conn->prepare("SELECT id, password FROM users WHERE email = ?");
        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }
        
        $stmt->bind_param("s", $email);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 0) {
            echo json_encode([
                'success' => false, 
                'message' => 'User not found',
                'error' => 'USER_NOT_FOUND'
            ]);
            $stmt->close();
            return;
        }

        $user = $result->fetch_assoc();
        $stmt->close();

        if (!password_verify($currentPassword, $user['password'])) {
            echo json_encode([
                'success' => false, 
                'message' => 'Current password is incorrect',
                'error' => 'INVALID_PASSWORD'
            ]);
            return;
        }

        // Hash new password and update
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        $stmt = $conn->prepare("UPDATE users SET password = ? WHERE email = ?");
        if (!$stmt) {
            throw new Exception('Prepare failed: ' . $conn->error);
        }
        
        $stmt->bind_param("ss", $hashedPassword, $email);
        $stmt->execute();
        $stmt->close();

        echo json_encode([
            'success' => true, 
            'message' => 'Password updated successfully'
        ]);
        
    } catch (Exception $e) {
        logError("Password update error: " . $e->getMessage());
        echo json_encode([
            'success' => false, 
            'message' => 'Password update failed',
            'error' => 'UPDATE_ERROR'
        ]);
    }
}
?>
