<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once 'config.php';

// Use PDO connection from config.php
try {
    $pdo = getDBConnection();
} catch (Exception $e) {
    die(json_encode(['success' => false, 'message' => 'Database connection failed: ' . $e->getMessage()]));
}

$input = json_decode(file_get_contents('php://input'), true);
$action = $input['action'] ?? $_GET['action'] ?? '';

switch ($action) {
    case 'login':
        handleLogin($pdo, $input);
        break;
    case 'register':
        handleRegister($pdo, $input);
        break;
    case 'get_user':
        handleGetUser($pdo, $input);
        break;
    case 'reset_password':
        handlePasswordReset($pdo, $input);
        break;
    case 'reset_confirm':
        handlePasswordResetConfirm($pdo, $input);
        break;
    case 'update_profile':
        handleProfileUpdate($pdo, $input);
        break;
    case 'update_password':
        handlePasswordUpdate($pdo, $input);
        break;
    default:
        echo json_encode(['success' => false, 'message' => 'Invalid action']);
}

function handleLogin($pdo, $input) {
    $email = $input['email'] ?? '';
    $password = $input['password'] ?? '';

    if (empty($email) || empty($password)) {
        echo json_encode(['success' => false, 'message' => 'Email and password are required']);
        return;
    }

    try {
        $stmt = $pdo->prepare("SELECT id, email, password, name, tokens, email_verified FROM users WHERE email = ?");
        $stmt->execute([$email]);
        $user = $stmt->fetch();

        if ($user && password_verify($password, $user['password'])) {
            // Remove password from response
            unset($user['password']);
            // Add email verification status from database
            $user['isEmailVerified'] = (bool)($user['email_verified'] ?? false);
            $user['email_verified'] = (bool)($user['email_verified'] ?? false);
            echo json_encode(['success' => true, 'user' => $user]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Invalid email or password']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Login failed: ' . $e->getMessage()]);
    }
}

function handleRegister($pdo, $input) {
    $email = $input['email'] ?? '';
    $password = $input['password'] ?? '';
    
    // Automatically extract username from email
    $name = $input['name'] ?? '';
    if (empty($name) && !empty($email)) {
        // Extract username from email (part before @)
        $name = explode('@', $email)[0];
        // Capitalize first letter
        $name = ucfirst(strtolower($name));
    }

    if (empty($email) || empty($password)) {
        echo json_encode(['success' => false, 'message' => 'Email and password are required']);
        return;
    }

    try {
        // Check if user already exists
        $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$email]);
        
        if ($stmt->fetch()) {
            echo json_encode(['success' => false, 'message' => 'User already exists']);
            return;
        }

        // Hash password
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        
        // Start transaction
        $pdo->beginTransaction();
        
        // Insert user (no free tokens - users must purchase)
        $stmt = $pdo->prepare("INSERT INTO users (email, password, name, tokens) VALUES (?, ?, ?, 0)");
        $stmt->execute([$email, $hashedPassword, $name]);
        $userId = $pdo->lastInsertId();

        // Create user settings
        $stmt = $pdo->prepare("INSERT INTO user_settings (user_email, special_offer_purchased, special_offer_shown) VALUES (?, 0, 0)");
        $stmt->execute([$email]);

        $pdo->commit();

        // Get user data for response
        $stmt = $pdo->prepare("SELECT id, email, name, tokens, email_verified FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();

        // Add email verification status from database
        $user['isEmailVerified'] = (bool)($user['email_verified'] ?? false);
        $user['email_verified'] = (bool)($user['email_verified'] ?? false);

        echo json_encode(['success' => true, 'user' => $user]);
    } catch (Exception $e) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Registration failed: ' . $e->getMessage()]);
    }
}

function handlePasswordReset($pdo, $input) {
    $email = $input['email'] ?? '';

    if (empty($email)) {
        echo json_encode(['success' => false, 'message' => 'Email is required']);
        return;
    }

    try {
        // Check if user exists
        $stmt = $pdo->prepare("SELECT id, name FROM users WHERE email = ?");
        $stmt->execute([$email]);
        $user = $stmt->fetch();
        
        if (!$user) {
            echo json_encode(['success' => false, 'message' => 'User not found']);
            return;
        }

        // Generate reset token
        $token = bin2hex(random_bytes(32));
        $expires = date('Y-m-d H:i:s', strtotime('+1 hour'));

        // Store reset token
        $stmt = $pdo->prepare("INSERT INTO password_resets (email, token, expires_at) VALUES (?, ?, ?) ON DUPLICATE KEY UPDATE token = ?, expires_at = ?");
        $stmt->execute([$email, $token, $expires, $token, $expires]);

        // Determine frontend URL
        $isLocalhost = $_SERVER['HTTP_HOST'] === 'localhost' || strpos($_SERVER['HTTP_HOST'], 'localhost') !== false;
        $frontendUrl = $isLocalhost ? 'http://localhost:5003' : 'https://app.tofeliq.com';
        $resetLink = $frontendUrl . '/?token=' . $token;

        // Send email
        $subject = "Reset Your Password - TOEFL Analyzer";
        $userName = $user['name'] ?? 'User';

        $emailBody = "
<!DOCTYPE html>
<html>
<head>
    <meta charset='UTF-8'>
    <style>
        body { font-family: Arial, sans-serif; background-color: #f4f7fa; margin: 0; padding: 0; }
        .email-container { max-width: 600px; margin: 40px auto; background: #ffffff; border-radius: 16px; overflow: hidden; box-shadow: 0 4px 20px rgba(0,0,0,0.1); }
        .email-header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 40px 30px; text-align: center; }
        .email-header h1 { color: #ffffff; margin: 0; font-size: 28px; }
        .email-body { padding: 40px 30px; }
        .reset-button { display: inline-block; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: #ffffff !important; text-decoration: none; padding: 16px 40px; border-radius: 8px; font-weight: 600; font-size: 16px; }
        .button-container { text-align: center; margin: 30px 0; }
        .warning { background: #fff3cd; padding: 15px; border-radius: 4px; border-left: 4px solid #f39c12; color: #856404; margin: 20px 0; }
    </style>
</head>
<body>
    <div class='email-container'>
        <div class='email-header'>
            <h1>🔐 TOEFL Analyzer</h1>
            <p style='color: rgba(255,255,255,0.9); margin: 10px 0 0;'>Password Reset Request</p>
        </div>
        <div class='email-body'>
            <h2 style='color: #2c3e50;'>Hello, $userName! 👋</h2>
            <p>We received a request to reset your password for your TOEFL Analyzer account.</p>
            <p>To reset your password, click the button below:</p>
            <div class='button-container'>
                <a href='$resetLink' class='reset-button'>🔑 Reset My Password</a>
            </div>
            <div class='warning'>
                <strong>⏰ Important:</strong> This link expires in 1 hour for security reasons.
            </div>
            <p><strong>If you didn't request this password reset, please ignore this email.</strong> Your password will remain unchanged.</p>
            <p>If the button doesn't work, copy this link into your browser:<br><a href='$resetLink'>$resetLink</a></p>
        </div>
        <div style='background: #f8f9fa; padding: 20px 30px; text-align: center; font-size: 13px; color: #7f8c8d;'>
            <p>&copy; " . date('Y') . " TOEFL Analyzer. All rights reserved.</p>
        </div>
    </div>
</body>
</html>
";

        $headers = "MIME-Version: 1.0" . "\r\n";
        $headers .= "Content-type: text/html; charset=UTF-8" . "\r\n";
        $headers .= "From: TOEFL Analyzer <noreply@tofeliq.com>" . "\r\n";

        $emailSent = mail($email, $subject, $emailBody, $headers);

        if ($emailSent) {
            echo json_encode([
                'success' => true, 
                'message' => 'Password reset link sent to your email. Please check your inbox.'
            ]);
        } else {
            error_log("Failed to send password reset email to: " . $email);
            echo json_encode([
                'success' => false,
                'message' => 'Failed to send password reset email. Please try again later.'
            ]);
        }
    } catch (Exception $e) {
        error_log("Password reset error: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Password reset failed: ' . $e->getMessage()]);
    }
}

function handlePasswordResetConfirm($pdo, $input) {
    $token = $input['token'] ?? '';
    $password = $input['password'] ?? '';

    if (empty($token) || empty($password)) {
        echo json_encode(['success' => false, 'message' => 'Token and password are required']);
        return;
    }

    try {
        // Check if token is valid and not expired
        $stmt = $pdo->prepare("SELECT email FROM password_resets WHERE token = ? AND expires_at > NOW()");
        $stmt->execute([$token]);
        $reset = $stmt->fetch();

        if (!$reset) {
            echo json_encode(['success' => false, 'message' => 'Invalid or expired token']);
            return;
        }

        $email = $reset['email'];

        // Update password
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE email = ?");
        $stmt->execute([$hashedPassword, $email]);

        // Delete token (one-time use - simpler than marking as used)
        $stmt = $pdo->prepare("DELETE FROM password_resets WHERE token = ?");
        $stmt->execute([$token]);

        echo json_encode(['success' => true, 'message' => 'Password updated successfully']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Password reset failed: ' . $e->getMessage()]);
    }
}

function handleProfileUpdate($pdo, $input) {
    $email = $input['email'] ?? '';
    $name = $input['name'] ?? '';
    $newEmail = $input['newEmail'] ?? '';

    if (empty($email)) {
        echo json_encode(['success' => false, 'message' => 'Current email is required']);
        return;
    }

    if (empty($name)) {
        echo json_encode(['success' => false, 'message' => 'Name is required']);
        return;
    }

    try {
        // Check if user exists
        $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
        $stmt->execute([$email]);
        
        if (!$stmt->fetch()) {
            echo json_encode(['success' => false, 'message' => 'User not found']);
            return;
        }

        // If email is being changed, check if new email already exists
        if (!empty($newEmail) && $newEmail !== $email) {
            $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
            $stmt->execute([$newEmail]);
            
            if ($stmt->fetch()) {
                echo json_encode(['success' => false, 'message' => 'Email already exists']);
                return;
            }
        }

        // Update user profile
        $updateEmail = !empty($newEmail) ? $newEmail : $email;
        $stmt = $pdo->prepare("UPDATE users SET name = ?, email = ? WHERE email = ?");
        $stmt->execute([$name, $updateEmail, $email]);

        // Get updated user data
        $stmt = $pdo->prepare("SELECT id, email, name, tokens, email_verified FROM users WHERE email = ?");
        $stmt->execute([$updateEmail]);
        $user = $stmt->fetch();

        // Add email verification status
        $user['isEmailVerified'] = (bool)($user['email_verified'] ?? false);
        $user['email_verified'] = (bool)($user['email_verified'] ?? false);

        echo json_encode(['success' => true, 'user' => $user, 'message' => 'Profile updated successfully']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Profile update failed: ' . $e->getMessage()]);
    }
}

function handlePasswordUpdate($pdo, $input) {
    $email = $input['email'] ?? '';
    $currentPassword = $input['currentPassword'] ?? '';
    $newPassword = $input['newPassword'] ?? '';

    // Log the request for debugging
    error_log("Password update request: " . json_encode([
        'email' => $email,
        'hasCurrentPassword' => !empty($currentPassword),
        'hasNewPassword' => !empty($newPassword)
    ]));

    if (empty($email) || empty($currentPassword) || empty($newPassword)) {
        echo json_encode(['success' => false, 'message' => 'Email, current password, and new password are required']);
        return;
    }

    try {
        // Check if user exists and verify current password
        $stmt = $pdo->prepare("SELECT id, password FROM users WHERE email = ?");
        $stmt->execute([$email]);
        $user = $stmt->fetch();

        if (!$user) {
            error_log("Password update failed: User not found for email: " . $email);
            echo json_encode(['success' => false, 'message' => 'User not found']);
            return;
        }

        if (!password_verify($currentPassword, $user['password'])) {
            error_log("Password update failed: Current password incorrect for email: " . $email);
            echo json_encode(['success' => false, 'message' => 'Current password is incorrect']);
            return;
        }

        // Hash new password and update
        $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE email = ?");
        $result = $stmt->execute([$hashedPassword, $email]);

        if ($result) {
            error_log("Password update successful for email: " . $email);
            echo json_encode(['success' => true, 'message' => 'Password updated successfully']);
        } else {
            error_log("Password update failed: Database update failed for email: " . $email);
            echo json_encode(['success' => false, 'message' => 'Password update failed']);
        }
    } catch (Exception $e) {
        error_log("Password update error: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Password update failed: ' . $e->getMessage()]);
    }
}

function handleGetUser($pdo, $input) {
    $email = $input['email'] ?? $_GET['email'] ?? '';

    if (empty($email)) {
        echo json_encode(['success' => false, 'message' => 'Email is required']);
        return;
    }

    try {
        // Get fresh user data from database
        $stmt = $pdo->prepare("SELECT id, email, name, tokens, email_verified, oauth_provider FROM users WHERE email = ?");
        $stmt->execute([$email]);
        $user = $stmt->fetch();

        if ($user) {
            // Add email verification status
            $user['isEmailVerified'] = (bool)($user['email_verified'] ?? false);
            $user['email_verified'] = (bool)($user['email_verified'] ?? false);
            
            echo json_encode(['success' => true, 'user' => $user]);
        } else {
            echo json_encode(['success' => false, 'message' => 'User not found']);
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Failed to get user data: ' . $e->getMessage()]);
    }
}
?> 