<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

require_once '../config.php';

try {
    $pdo = getDBConnection();
    $input = json_decode(file_get_contents('php://input'), true);
    $email = $input['email'] ?? null;
    $password = $input['password'] ?? null;
    $name = $input['name'] ?? null;
    $affiliateCode = $input['affiliateCode'] ?? null;
    
    // Auto-generate name from email if not provided
    if (empty($name) && !empty($email)) {
        $name = explode('@', $email)[0];
        $name = ucfirst(strtolower($name));
    }
    
    if (!$email || !$password || !$affiliateCode) {
        throw new Exception('Email, password, and affiliate code are required');
    }
    
    if (empty($name)) {
        throw new Exception('Unable to generate name from email');
    }
    
    // Validate affiliate code
    $stmt = $pdo->prepare("SELECT user_id FROM affiliate_data WHERE affiliate_code = ?");
    $stmt->execute([$affiliateCode]);
    $affiliateUser = $stmt->fetch();
    
    if (!$affiliateUser) {
        throw new Exception('Invalid affiliate code');
    }
    
    $affiliateUserId = $affiliateUser['user_id'];
    
    // Check if email already exists
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        throw new Exception('Email already registered');
    }
    
    // Start transaction
    $pdo->beginTransaction();
    
    try {
        // Create new user (no free tokens - must purchase)
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("
            INSERT INTO users (email, password, name, created_at, tokens)
            VALUES (?, ?, ?, NOW(), 0)
        ");
        $stmt->execute([$email, $hashedPassword, $name]);
        $newUserId = $pdo->lastInsertId();
        
        // Create referral record
        $stmt = $pdo->prepare("
            INSERT INTO referrals (affiliate_id, referred_user_id, referred_email, referred_name, status, created_at, earnings)
            VALUES (?, ?, ?, ?, 'active', NOW(), 0)
        ");
        $stmt->execute([$affiliateUserId, $newUserId, $email, $name]);
        
        // Update affiliate statistics
        $stmt = $pdo->prepare("
            UPDATE affiliate_data 
            SET total_referrals = total_referrals + 1
            WHERE user_id = ?
        ");
        $stmt->execute([$affiliateUserId]);
        
        $pdo->commit();
        
        // Get user data for response
        $stmt = $pdo->prepare("SELECT id, email, name, tokens, email_verified FROM users WHERE id = ?");
        $stmt->execute([$newUserId]);
        $user = $stmt->fetch();
        
        // Add email verification status
        $user['isEmailVerified'] = (bool)($user['email_verified'] ?? false);
        $user['email_verified'] = (bool)($user['email_verified'] ?? false);
        
        echo json_encode([
            'success' => true,
            'userId' => $newUserId,
            'user' => $user,
            'message' => 'Registration successful with affiliate code'
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
?>
